import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "npm:@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "*",
};

interface OperaCredentials {
  hotelId: string;
  apiUrl: string;
  username: string;
  password: string;
  propertyCode: string;
}

interface MigrationRequest {
  credentials: OperaCredentials;
  dataTypes: string[]; // ['guests', 'bookings', 'rooms']
}

serve(async (req) => {
  const requestId = crypto.randomUUID();
  console.log(`[${requestId}] Opera migration request received`);

  // Handle CORS preflight
  if (req.method === "OPTIONS") {
    return new Response(null, { status: 204, headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    const body = await req.json() as MigrationRequest;
    const { credentials, dataTypes } = body;

    console.log(`[${requestId}] Starting Opera migration for hotel: ${credentials.hotelId}`);

    // Authenticate with Opera PMS
    const operaAuth = await authenticateOpera(credentials);
    if (!operaAuth.success) {
      throw new Error(`Opera authentication failed: ${operaAuth.error}`);
    }

    const results = {
      guests: 0,
      bookings: 0,
      rooms: 0,
      errors: [] as string[],
    };

    // Migrate Guests
    if (dataTypes.includes("guests")) {
      console.log(`[${requestId}] Migrating guests...`);
      const guestsResult = await migrateGuests(credentials, operaAuth.token, supabase, credentials.hotelId);
      results.guests = guestsResult.count;
      if (guestsResult.errors.length > 0) {
        results.errors.push(...guestsResult.errors);
      }
    }

    // Migrate Bookings
    if (dataTypes.includes("bookings")) {
      console.log(`[${requestId}] Migrating bookings...`);
      const bookingsResult = await migrateBookings(credentials, operaAuth.token, supabase, credentials.hotelId);
      results.bookings = bookingsResult.count;
      if (bookingsResult.errors.length > 0) {
        results.errors.push(...bookingsResult.errors);
      }
    }

    // Migrate Rooms
    if (dataTypes.includes("rooms")) {
      console.log(`[${requestId}] Migrating rooms...`);
      const roomsResult = await migrateRooms(credentials, operaAuth.token, supabase, credentials.hotelId);
      results.rooms = roomsResult.count;
      if (roomsResult.errors.length > 0) {
        results.errors.push(...roomsResult.errors);
      }
    }

    console.log(`[${requestId}] Migration completed:`, results);

    return new Response(
      JSON.stringify({
        success: true,
        results,
        message: `Successfully migrated ${results.guests} guests, ${results.bookings} bookings, and ${results.rooms} rooms`,
      }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );

  } catch (error) {
    console.error(`[${requestId}] Migration error:`, error);
    return new Response(
      JSON.stringify({
        success: false,
        error: error.message,
      }),
      { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});

async function authenticateOpera(credentials: OperaCredentials) {
  try {
    // Opera PMS uses Basic Authentication
    const authString = btoa(`${credentials.username}:${credentials.password}`);
    
    const response = await fetch(`${credentials.apiUrl}/api/v1/auth`, {
      method: "POST",
      headers: {
        "Authorization": `Basic ${authString}`,
        "Content-Type": "application/json",
        "x-app-key": credentials.propertyCode,
      },
    });

    if (!response.ok) {
      return { success: false, error: `Authentication failed: ${response.statusText}` };
    }

    const data = await response.json();
    return { success: true, token: data.access_token };

  } catch (error) {
    return { success: false, error: error.message };
  }
}

async function migrateGuests(credentials: OperaCredentials, token: string, supabase: any, hotelId: string) {
  const errors: string[] = [];
  let count = 0;

  try {
    // Fetch guests from Opera PMS
    const response = await fetch(`${credentials.apiUrl}/api/v1/profiles`, {
      headers: {
        "Authorization": `Bearer ${token}`,
        "x-app-key": credentials.propertyCode,
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch guests: ${response.statusText}`);
    }

    const operaGuests = await response.json();

    // Transform and insert into Supabase
    for (const guest of operaGuests.profiles || []) {
      try {
        const guestData = {
          hotel_id: hotelId,
          first_name: guest.name?.firstName || "",
          last_name: guest.name?.lastName || "",
          email: guest.emails?.[0]?.email || "",
          phone: guest.phones?.[0]?.number || "",
          address: guest.addresses?.[0]?.address || "",
          city: guest.addresses?.[0]?.city || "",
          country: guest.addresses?.[0]?.country || "",
          vip_status: guest.vipStatus || false,
          preferences: guest.preferences || {},
          external_id: guest.profileId,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${hotelId}_guests`)
          .insert(guestData);

        if (error) {
          errors.push(`Guest ${guest.name?.firstName} ${guest.name?.lastName}: ${error.message}`);
        } else {
          count++;
        }
      } catch (err) {
        errors.push(`Guest processing error: ${err.message}`);
      }
    }

  } catch (error) {
    errors.push(`Guests migration error: ${error.message}`);
  }

  return { count, errors };
}

async function migrateBookings(credentials: OperaCredentials, token: string, supabase: any, hotelId: string) {
  const errors: string[] = [];
  let count = 0;

  try {
    // Fetch reservations from Opera PMS
    const response = await fetch(`${credentials.apiUrl}/api/v1/reservations`, {
      headers: {
        "Authorization": `Bearer ${token}`,
        "x-app-key": credentials.propertyCode,
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch bookings: ${response.statusText}`);
    }

    const operaBookings = await response.json();

    // Transform and insert into Supabase
    for (const booking of operaBookings.reservations || []) {
      try {
        const bookingData = {
          hotel_id: hotelId,
          guest_name: `${booking.guestName?.firstName || ""} ${booking.guestName?.lastName || ""}`,
          guest_email: booking.guestEmail || "",
          guest_phone: booking.guestPhone || "",
          room_type: booking.roomType || "",
          room_number: booking.roomNumber || "",
          check_in: booking.arrivalDate,
          check_out: booking.departureDate,
          adults: booking.adults || 1,
          children: booking.children || 0,
          total_amount: booking.totalAmount || 0,
          status: booking.status || "confirmed",
          external_id: booking.reservationId,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${hotelId}_bookings`)
          .insert(bookingData);

        if (error) {
          errors.push(`Booking ${booking.reservationId}: ${error.message}`);
        } else {
          count++;
        }
      } catch (err) {
        errors.push(`Booking processing error: ${err.message}`);
      }
    }

  } catch (error) {
    errors.push(`Bookings migration error: ${error.message}`);
  }

  return { count, errors };
}

async function migrateRooms(credentials: OperaCredentials, token: string, supabase: any, hotelId: string) {
  const errors: string[] = [];
  let count = 0;

  try {
    // Fetch rooms from Opera PMS
    const response = await fetch(`${credentials.apiUrl}/api/v1/rooms`, {
      headers: {
        "Authorization": `Bearer ${token}`,
        "x-app-key": credentials.propertyCode,
      },
    });

    if (!response.ok) {
      throw new Error(`Failed to fetch rooms: ${response.statusText}`);
    }

    const operaRooms = await response.json();

    // Transform and insert into Supabase
    for (const room of operaRooms.rooms || []) {
      try {
        const roomData = {
          hotel_id: hotelId,
          room_number: room.roomNumber,
          room_type: room.roomType,
          floor: room.floor || 1,
          capacity: room.maxOccupancy || 2,
          price: room.baseRate || 0,
          amenities: room.amenities || [],
          status: room.status || "available",
          external_id: room.roomId,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${hotelId}_rooms`)
          .insert(roomData);

        if (error) {
          errors.push(`Room ${room.roomNumber}: ${error.message}`);
        } else {
          count++;
        }
      } catch (err) {
        errors.push(`Room processing error: ${err.message}`);
      }
    }

  } catch (error) {
    errors.push(`Rooms migration error: ${error.message}`);
  }

  return { count, errors };
}