import { serve } from "https://deno.land/std@0.168.0/http/server.ts";
import { createClient } from "npm:@supabase/supabase-js@2";

const corsHeaders = {
  "Access-Control-Allow-Origin": "*",
  "Access-Control-Allow-Headers": "*",
};

interface CloudbedsCredentials {
  hotelId: string;
  propertyId: string;
  clientId: string;
  clientSecret: string;
}

serve(async (req) => {
  const requestId = crypto.randomUUID();
  console.log(`[${requestId}] Cloudbeds migration request received`);

  if (req.method === "OPTIONS") {
    return new Response(null, { status: 204, headers: corsHeaders });
  }

  try {
    const supabaseUrl = Deno.env.get("SUPABASE_URL")!;
    const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY")!;
    const supabase = createClient(supabaseUrl, supabaseKey);

    const body = await req.json();
    const { credentials, dataTypes } = body;

    console.log(`[${requestId}] Starting Cloudbeds migration for hotel: ${credentials.hotelId}`);

    // Get Cloudbeds access token
    const token = await getCloudbedsToken(credentials);
    if (!token) {
      throw new Error("Failed to authenticate with Cloudbeds");
    }

    const results = {
      guests: 0,
      bookings: 0,
      rooms: 0,
      errors: [] as string[],
    };

    const cloudbedsApiUrl = "https://api.cloudbeds.com/api/v1.1";

    if (dataTypes.includes("guests")) {
      const guestsResult = await migrateGuestsFromCloudbeds(credentials, token, cloudbedsApiUrl, supabase);
      results.guests = guestsResult.count;
      results.errors.push(...guestsResult.errors);
    }

    if (dataTypes.includes("bookings")) {
      const bookingsResult = await migrateBookingsFromCloudbeds(credentials, token, cloudbedsApiUrl, supabase);
      results.bookings = bookingsResult.count;
      results.errors.push(...bookingsResult.errors);
    }

    if (dataTypes.includes("rooms")) {
      const roomsResult = await migrateRoomsFromCloudbeds(credentials, token, cloudbedsApiUrl, supabase);
      results.rooms = roomsResult.count;
      results.errors.push(...roomsResult.errors);
    }

    return new Response(
      JSON.stringify({
        success: true,
        results,
        message: `Successfully migrated ${results.guests} guests, ${results.bookings} bookings, and ${results.rooms} rooms from Cloudbeds`,
      }),
      { headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );

  } catch (error) {
    console.error(`[${requestId}] Migration error:`, error);
    return new Response(
      JSON.stringify({ success: false, error: error.message }),
      { status: 400, headers: { ...corsHeaders, "Content-Type": "application/json" } }
    );
  }
});

async function getCloudbedsToken(credentials: CloudbedsCredentials) {
  try {
    const response = await fetch("https://api.cloudbeds.com/api/v1.1/oauth/token", {
      method: "POST",
      headers: { "Content-Type": "application/x-www-form-urlencoded" },
      body: new URLSearchParams({
        grant_type: "client_credentials",
        client_id: credentials.clientId,
        client_secret: credentials.clientSecret,
      }),
    });

    if (!response.ok) return null;
    
    const data = await response.json();
    return data.access_token;
  } catch {
    return null;
  }
}

async function migrateGuestsFromCloudbeds(credentials: CloudbedsCredentials, token: string, apiUrl: string, supabase: any) {
  const errors: string[] = [];
  let count = 0;

  try {
    const response = await fetch(`${apiUrl}/getGuests?propertyID=${credentials.propertyId}`, {
      headers: { "Authorization": `Bearer ${token}` },
    });

    if (!response.ok) throw new Error(`Cloudbeds API error: ${response.statusText}`);

    const data = await response.json();

    for (const guest of data.data || []) {
      try {
        const guestData = {
          hotel_id: credentials.hotelId,
          first_name: guest.guestFirstName || "",
          last_name: guest.guestLastName || "",
          email: guest.guestEmail || "",
          phone: guest.guestPhone || "",
          address: guest.guestAddress || "",
          city: guest.guestCity || "",
          country: guest.guestCountry || "",
          external_id: guest.guestID,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${credentials.hotelId}_guests`)
          .insert(guestData);

        if (!error) count++;
        else errors.push(`Guest ${guest.guestFirstName}: ${error.message}`);
      } catch (err) {
        errors.push(`Guest processing error: ${err.message}`);
      }
    }
  } catch (error) {
    errors.push(`Cloudbeds guests migration error: ${error.message}`);
  }

  return { count, errors };
}

async function migrateBookingsFromCloudbeds(credentials: CloudbedsCredentials, token: string, apiUrl: string, supabase: any) {
  const errors: string[] = [];
  let count = 0;

  try {
    const response = await fetch(`${apiUrl}/getReservations?propertyID=${credentials.propertyId}`, {
      headers: { "Authorization": `Bearer ${token}` },
    });

    if (!response.ok) throw new Error(`Cloudbeds API error: ${response.statusText}`);

    const data = await response.json();

    for (const reservation of data.data || []) {
      try {
        const bookingData = {
          hotel_id: credentials.hotelId,
          guest_name: `${reservation.guestFirstName || ""} ${reservation.guestLastName || ""}`,
          guest_email: reservation.guestEmail || "",
          guest_phone: reservation.guestPhone || "",
          room_type: reservation.roomTypeName || "",
          room_number: reservation.roomName || "",
          check_in: reservation.startDate,
          check_out: reservation.endDate,
          adults: reservation.adults || 1,
          children: reservation.children || 0,
          total_amount: reservation.grandTotal || 0,
          status: reservation.status || "confirmed",
          external_id: reservation.reservationID,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${credentials.hotelId}_bookings`)
          .insert(bookingData);

        if (!error) count++;
        else errors.push(`Booking ${reservation.reservationID}: ${error.message}`);
      } catch (err) {
        errors.push(`Booking processing error: ${err.message}`);
      }
    }
  } catch (error) {
    errors.push(`Cloudbeds bookings migration error: ${error.message}`);
  }

  return { count, errors };
}

async function migrateRoomsFromCloudbeds(credentials: CloudbedsCredentials, token: string, apiUrl: string, supabase: any) {
  const errors: string[] = [];
  let count = 0;

  try {
    const response = await fetch(`${apiUrl}/getRooms?propertyID=${credentials.propertyId}`, {
      headers: { "Authorization": `Bearer ${token}` },
    });

    if (!response.ok) throw new Error(`Cloudbeds API error: ${response.statusText}`);

    const data = await response.json();

    for (const room of data.data || []) {
      try {
        const roomData = {
          hotel_id: credentials.hotelId,
          room_number: room.roomName,
          room_type: room.roomTypeName || "standard",
          capacity: room.maxGuests || 2,
          status: room.roomStatus || "available",
          external_id: room.roomID,
          created_at: new Date().toISOString(),
        };

        const { error } = await supabase
          .from(`app_${credentials.hotelId}_rooms`)
          .insert(roomData);

        if (!error) count++;
        else errors.push(`Room ${room.roomName}: ${error.message}`);
      } catch (err) {
        errors.push(`Room processing error: ${err.message}`);
      }
    }
  } catch (error) {
    errors.push(`Cloudbeds rooms migration error: ${error.message}`);
  }

  return { count, errors };
}