import { serve } from 'https://deno.land/std@0.168.0/http/server.ts'

const corsHeaders = {
  'Access-Control-Allow-Origin': '*',
  'Access-Control-Allow-Headers': 'authorization, x-client-info, apikey, content-type',
}

serve(async (req) => {
  // Handle CORS preflight
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders })
  }

  try {
    const { check } = await req.json()
    
    // Extract hotel ID from the function URL
    const url = new URL(req.url)
    const pathParts = url.pathname.split('/')
    const functionName = pathParts[pathParts.length - 1]
    const hotelIdMatch = functionName.match(/app_([a-zA-Z0-9]+)_check_config/)
    
    if (!hotelIdMatch) {
      throw new Error('Invalid function name format')
    }
    
    const hotelId = hotelIdMatch[1]
    
    let configured = false
    
    if (check === 'openai') {
      // Check if OpenAI API key is configured for this hotel
      const openaiKey = Deno.env.get(`APP_${hotelId.toUpperCase()}_OPENAI_API_KEY`)
      configured = !!openaiKey && openaiKey.startsWith('sk-')
    }
    
    return new Response(
      JSON.stringify({ configured }),
      {
        headers: { ...corsHeaders, 'Content-Type': 'application/json' }
      }
    )
    
  } catch (error) {
    console.error('Check Config Error:', error)
    return new Response(
      JSON.stringify({ configured: false, error: error.message }),
      {
        status: 500,
        headers: { ...corsHeaders, 'Content-Type': 'application/json' }
      }
    )
  }
})